<?php
/**
 * ================================================================
 * DRIPBOX LASSO IMPORTER - MAIN LOADER
 * ================================================================
 * Initializes the Lasso Importer module
 * ================================================================
 */

if (!defined('ABSPATH')) {
    exit;
}

class DripBox_Lasso_Importer {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }
    
    /**
     * Load required files
     */
    private function load_dependencies() {
        // AJAX handlers
        require_once plugin_dir_path(__FILE__) . 'class-ajax.php';
        
        // Admin page (only in admin)
        if (is_admin()) {
            require_once plugin_dir_path(__FILE__) . 'admin/class-admin-page.php';
        }
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        if (!is_admin()) {
            return;
        }
        
        add_action('admin_menu', [$this, 'register_admin_menu']);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_assets']);
    }
    
    /**
     * Register admin menu
     */
    public function register_admin_menu() {
        add_submenu_page(
            'edit.php?post_type=dripbox_product',
            'Lasso Importer',
            'Lasso Importer',
            'manage_options',
            'dripbox-lasso-importer',
            [$this, 'render_admin_page']
        );
    }
    
    /**
     * Render admin page
     */
    public function render_admin_page() {
        $admin_page = new DripBox_Lasso_Importer_Admin();
        $admin_page->render();
    }
    
    /**
     * Enqueue CSS/JS
     */
    public function enqueue_assets($hook) {
        // Only load on importer page
        if ($hook !== 'dripbox_product_page_dripbox-lasso-importer') {
            return;
        }
        
        // CSS
        wp_enqueue_style(
            'dripbox-importer',
            plugins_url('assets/importer.css', __FILE__),
            [],
            '1.0.0'
        );
        
        // JS
        wp_enqueue_script(
            'dripbox-importer',
            plugins_url('assets/importer.js', __FILE__),
            ['jquery'],
            '1.0.0',
            true
        );
        
        // Pass data to JS
        wp_localize_script('dripbox-importer', 'dripboxImporter', [
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('dripbox_importer')
        ]);
    }
    
    /**
     * Detect if Lasso is installed
     */
    public function detect_lasso() {
        global $wpdb;
        
        $count = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$wpdb->posts} 
            WHERE post_type = 'lasso_post'
            AND post_status = 'publish'
        ");
        
        return $count > 0;
    }
    
    /**
     * Count Lasso products
     */
    public function count_lasso_products() {
        global $wpdb;
        
        return $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$wpdb->posts} 
            WHERE post_type = 'lasso_post'
            AND post_status = 'publish'
        ");
    }
    
    /**
     * Collect all Lasso data (RAW)
     */
    public function collect_lasso_data() {
        global $wpdb;
        
        $lasso_products = $wpdb->get_results("
            SELECT p.ID, p.post_title, p.post_content
            FROM {$wpdb->posts} p
            WHERE p.post_type = 'lasso_post'
            AND p.post_status = 'publish'
        ");
        
        $data = [];
        
        foreach ($lasso_products as $product) {
           // Get ONLY Lasso meta data (filtered)
            $meta = $wpdb->get_results($wpdb->prepare("
                SELECT meta_key, meta_value
                FROM {$wpdb->postmeta}
                WHERE post_id = %d
                AND meta_key LIKE 'lasso_%%'
            ", $product->ID), ARRAY_A);;
            
            $data[] = [
                'id' => $product->ID,
                'title' => $product->post_title,
                'content' => $product->post_content,
                'meta' => $meta
            ];
        }
        
        return $data;
    }
}

// Initialize
function dripbox_lasso_importer_init() {
    DripBox_Lasso_Importer::get_instance();
}
add_action('plugins_loaded', 'dripbox_lasso_importer_init');