<?php
/**
 * ================================================================
 * DRIPBOX LINK CHECKER - DATABASE HANDLER
 * ================================================================
 * Purpose: Save and retrieve link check results
 * Storage: wp_options (simple key-value)
 * ================================================================
 */

if (!defined('ABSPATH')) {
    exit;
}

class DripBox_Link_Checker_DB {
    
    private static $instance = null;
    private $option_key = 'dripbox_link_checker_results';
    
    /**
     * Singleton
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Save all results
     */
    public function save_results($results) {
        $data = [
            'results' => $results,
            'total_links' => count($results),
            'broken_count' => $this->count_by_status($results, 'broken'),
            'risky_count' => $this->count_by_status($results, 'risky'),
            'ok_count' => $this->count_by_status($results, 'ok'),
            'last_scan' => current_time('mysql'),
            'scan_duration' => 0 // Will be updated by scanner
        ];
        
        update_option($this->option_key, $data, false);
    }
    
    /**
     * Get all results
     */
    public function get_results() {
        $data = get_option($this->option_key, []);
        
        // Return default structure if empty
        if (empty($data)) {
            return [
                'results' => [],
                'total_links' => 0,
                'broken_count' => 0,
                'risky_count' => 0,
                'ok_count' => 0,
                'last_scan' => null,
                'scan_duration' => 0
            ];
        }
        
        return $data;
    }
    
    /**
     * Clear all results
     */
    public function clear_results() {
        delete_option($this->option_key);
    }
    
    /**
     * Update scan duration
     */
    public function update_scan_duration($seconds) {
        $data = $this->get_results();
        $data['scan_duration'] = $seconds;
        update_option($this->option_key, $data, false);
    }
    
    /**
     * Count by status
     */
    private function count_by_status($results, $status) {
        return count(array_filter($results, function($item) use ($status) {
            return isset($item['status']) && $item['status'] === $status;
        }));
    }
    
    /**
     * Get broken links only
     */
    public function get_broken_links() {
        $data = $this->get_results();
        return array_filter($data['results'], function($item) {
            return isset($item['status']) && $item['status'] === 'broken';
        });
    }
    
    /**
     * Get risky links only
     */
    public function get_risky_links() {
        $data = $this->get_results();
        return array_filter($data['results'], function($item) {
            return isset($item['status']) && $item['status'] === 'risky';
        });
    }
}