<?php
if (!defined('ABSPATH')) exit;

class Dripbox_Update_Checker {

    private $api_url = 'https://getdripbox.com/dripbox-updates/update-server.php';
    private $plugin_file;
    private $plugin_slug;

    public function __construct($plugin_file) {
        $this->plugin_file = $plugin_file;
        $this->plugin_slug = plugin_basename($plugin_file); // Ex: dripbox/dripbox.php

        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_update']);
        add_filter('plugins_api', [$this, 'plugin_info'], 10, 3);
    }

    public function check_for_update($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        // 1. Consulta sua API
        $response = wp_remote_get($this->api_url, [
            'timeout' => 5, // Não trava o site se sua API estiver lenta
            'headers' => ['Accept' => 'application/json']
        ]);

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return $transient;
        }

        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body);

        // Se a API não devolver JSON válido, aborta
        if (!$data || !isset($data->version)) {
            return $transient;
        }

        // 2. Compara versões
        $current_version = $transient->checked[$this->plugin_slug] ?? '0.0.0';

        if (version_compare($data->version, $current_version, '>')) {
            // PROTEÇÃO: Só injeta a atualização no slot do DripBox
            $obj = new stdClass();
            $obj->slug = 'dripbox'; // Slug simples
            $obj->plugin = $this->plugin_slug; // Slug completo (pasta/arquivo)
            $obj->new_version = $data->version;
            $obj->package = $data->download_url;
            $obj->url = 'https://getdripbox.com';
            
            // Ícones e Banners (Opcional, deixa mais bonito na lista)
            $obj->icons = [
                '1x' => 'https://getdripbox.com/wp-content/uploads/icon-128x128.png',
                '2x' => 'https://getdripbox.com/wp-content/uploads/icon-256x256.png'
            ];

            $transient->response[$this->plugin_slug] = $obj;
        }

        return $transient;
    }

    public function plugin_info($false, $action, $args) {
        // PROTEÇÃO 1: Só roda se a ação for pedir info de plugin
        if ($action !== 'plugin_information') {
            return $false;
        }

        // PROTEÇÃO 2 (A MAIS IMPORTANTE): Só responde se perguntarem pelo "dripbox"
        // O código antigo não tinha isso, por isso sequestrava o "wp-code"
        if (!isset($args->slug) || $args->slug !== 'dripbox') {
            return $false;
        }

        // Se chegou aqui, o WP está perguntando: "Quem é DripBox?"
        $response = wp_remote_get($this->api_url, ['timeout' => 5]);

        if (is_wp_error($response)) {
            return $false;
        }

        $data = json_decode(wp_remote_retrieve_body($response));

        if (!$data) {
            return $false;
        }

        // Monta a ficha técnica (Popup de detalhes)
        $info = new stdClass();
        $info->name = 'DripBox Affiliate';
        $info->slug = 'dripbox';
        $info->version = $data->version;
        $info->author = '<a href="https://getdripbox.com">Rafa de Atlanta</a>';
        $info->author_profile = 'https://getdripbox.com';
        $info->download_link = $data->download_url;
        $info->trunk = $data->download_url;
        $info->last_updated = $data->last_updated ?? date('Y-m-d H:i:s');
        $info->homepage = 'https://getdripbox.com';
        $info->sections = [
            'description' => 'Smart affiliate boxes with clean design.',
            'changelog'   => $data->changelog ?? 'New version available.'
        ];
        
        // Banner do popup (opcional)
        $info->banners = [
            'low'  => 'https://getdripbox.com/wp-content/uploads/banner-772x250.jpg',
            'high' => 'https://getdripbox.com/wp-content/uploads/banner-1544x500.jpg'
        ];

        return $info;
    }
}